-- output of CoreGen module generator
-- $RCSfile: sectionb.vhd,v $ $Revision: 1.1 $
-- ************************************************************************
--  Copyright (C) 1996 - Xilinx, Inc.
-- ************************************************************************
--
--  Description:
--    Single channel SDA FIR filter behavioral model
--

LIBRARY ieee;
USE ieee.std_logic_1164.ALL;

PACKAGE sdafirbehv_pack IS

-- ------------------------------------------------------------------------ --
-- TYPE DECLARATIONS                                                        --
-- ------------------------------------------------------------------------ --

  TYPE coeficient_list IS ARRAY(0 TO 39) OF INTEGER;

  TYPE filter_latency_type IS ARRAY(1 TO 80) OF INTEGER;
  TYPE filter_latency_table_type IS ARRAY(0 TO 1) OF filter_latency_type;

  TYPE filter_growth_type IS ARRAY(1 TO 80) OF INTEGER;
  TYPE growth_table_type IS ARRAY(0 TO 1) OF filter_growth_type;

-- ------------------------------------------------------------------------ --
-- CONSTANT DECLARATIONS                                                    --
-- ------------------------------------------------------------------------ --

  -- Table appropriate for tall-thin symmetric filters and non-symmetric    --
  -- filters short-wide symmetric filters need an adjustment                --

  CONSTANT latency_table : filter_latency_table_type := ( 0 =>
                               ( 3 => 3, 4  => 3, 5  => 3, 6  => 3, 7  => 3, 8  => 3, 9  => 3,
     10 => 3, 11 => 4, 12 => 4, 13 => 4, 14 => 4, 15 => 4, 16 => 4, 17 => 4, 18 => 4, 19 => 4,
     20 => 4, 21 => 5, 22 => 5, 23 => 5, 24 => 5, 25 => 5, 26 => 5, 27 => 5, 28 => 5, 29 => 5,
     30 => 5, 31 => 5, 32 => 5, 33 => 5, 34 => 5, 35 => 5, 36 => 5, 37 => 5, 38 => 5, 39 => 5,
     40 => 5, OTHERS => 0 ),
                                                          1 =>
                               ( 3 => 4, 4  => 4, 5  => 4, 6  => 4, 7  => 4, 8  => 4, 9  => 4,
     10 => 4, 11 => 4, 12 => 4, 13 => 4, 14 => 4, 15 => 4, 16 => 4, 17 => 4, 18 => 4, 19 => 4,
     20 => 4, 21 => 5, 22 => 5, 23 => 5, 24 => 5, 25 => 5, 26 => 5, 27 => 5, 28 => 5, 29 => 5,
     30 => 5, 31 => 5, 32 => 5, 33 => 5, 34 => 5, 35 => 5, 36 => 5, 37 => 5, 38 => 6, 39 => 5,
     40 => 5, 41 => 6, 42 => 6, 43 => 6, 44 => 6, 45 => 6, 46 => 6, 47 => 6, 48 => 6, 49 => 6,
     50 => 6, 51 => 6, 52 => 6, 53 => 6, 54 => 6, 55 => 6, 56 => 6, 57 => 6, 58 => 6, 59 => 6,
     60 => 6, 61 => 6, 62 => 6, 63 => 6, 64 => 6, 65 => 6, 66 => 6, 67 => 6, 68 => 6, 69 => 6,
     70 => 6, 71 => 6, 72 => 6, 73 => 6, 74 => 6, 75 => 6, 76 => 6, 77 => 6, 78 => 6, 79 => 6,
     80 => 6, OTHERS => 0 ) );

  -- Table appropriate for tall-thin symmetric filters and non-symmetric   --
  -- filters, short-wide symmetric filters need an adjustment              --

  CONSTANT growth_table : growth_table_type := ( 0 =>
                               ( 3 => 1, 4  => 1, 5  => 2, 6  => 3, 7  => 3, 8  => 3, 9  => 4,
     10 => 4, 11 => 4, 12 => 4, 13 => 5, 14 => 5, 15 => 4, 16 => 4, 17 => 5, 18 => 5, 19 => 5,
     20 => 5, 21 => 5, 22 => 6, 23 => 5, 24 => 5, 25 => 5, 26 => 5, 27 => 5, 28 => 5, 29 => 6,
     30 => 6, 31 => 5, 32 => 5, 33 => 6, 34 => 6, 35 => 6, 36 => 6, 37 => 6, 38 => 6, 39 => 6,
     40 => 6, OTHERS => 0 ),
                                                 1 =>
                               ( 3 => 2, 4  => 2, 5  => 2, 6  => 2, 7  => 2, 8  => 2, 9  => 3,
     10 => 3, 11 => 3, 12 => 3, 13 => 3, 14 => 3, 15 => 3, 16 => 3, 17 => 4, 18 => 4, 19 => 4,
     20 => 4, 21 => 4, 22 => 4, 23 => 4, 24 => 4, 25 => 5, 26 => 5, 27 => 4, 28 => 4, 29 => 4,
     30 => 4, 31 => 4, 32 => 4, 33 => 5, 34 => 5, 35 => 5, 36 => 5, 37 => 5, 38 => 5, 39 => 5,
     40 => 5, 41 => 5, 42 => 5, 43 => 5, 44 => 5, 45 => 5, 46 => 5, 47 => 5, 48 => 5, 49 => 6,
     50 => 6, 51 => 6, 52 => 6, 53 => 6, 54 => 6, 55 => 6, 56 => 6, 57 => 5, 58 => 5, 59 => 6,
     60 => 6, 61 => 5, 62 => 5, 63 => 5, 64 => 5, 65 => 6, 66 => 6, 67 => 6, 68 => 6, 69 => 6,
     70 => 6, 71 => 6, 72 => 6, 73 => 6, 74 => 6, 75 => 6, 76 => 6, 77 => 6, 78 => 6, 79 => 6,
     80 => 6, OTHERS => 0 ));

-- ------------------------------------------------------------------------ --
-- FUNCTION PROTOTYPES                                                      --
-- ------------------------------------------------------------------------ --

  FUNCTION eval( a : BOOLEAN )
    RETURN INTEGER;

  FUNCTION rat( a : std_logic )
    RETURN std_logic;

  FUNCTION rat_v( a : std_logic_vector )
    RETURN std_logic_vector;

  FUNCTION div2roundup( a : INTEGER )
    RETURN INTEGER;

  FUNCTION range_check( value, lower_limit, upper_limit : INTEGER )
    RETURN INTEGER;

  FUNCTION check_coefs( coeficients : coeficient_list; number_of_coeficient_bits : INTEGER )
    RETURN coeficient_list;

END sdafirbehv_pack;

-- ------------------------------------------------------------------------ --
-- FUNCTIONS                                                                --
-- ------------------------------------------------------------------------ --

PACKAGE BODY sdafirbehv_pack IS

  FUNCTION eval( a : BOOLEAN )
    RETURN INTEGER IS
  BEGIN
    IF a THEN
      RETURN 1;
    ELSE
      RETURN 0;
    END IF;
  END eval;

-- ------------------------------------------------------------------------ --

  FUNCTION rat( a : std_logic )
    RETURN std_logic IS

    VARIABLE result : std_logic;

  BEGIN

    IF a='1' OR a='H' THEN
      result := '1';
    ELSIF a='0' OR a='L' THEN
      result := '0';
    ELSE
      result := 'X';
    END IF;
    RETURN result;

  END rat;

-- ------------------------------------------------------------------------ --

  FUNCTION rat_v( a : std_logic_vector )
    RETURN std_logic_vector IS

    VARIABLE result : std_logic_vector(a'RANGE);

  BEGIN

    FOR i IN a'RANGE LOOP
      result(i) := rat(a(i));
    END LOOP;
    RETURN result;

  END rat_v;

-- ------------------------------------------------------------------------ --

  FUNCTION div2roundup( a : INTEGER )
    RETURN INTEGER IS
  BEGIN
    RETURN (a / 2) + (a MOD 2);
  END div2roundup;

-- ------------------------------------------------------------------------ --

  FUNCTION range_check( value, lower_limit, upper_limit : INTEGER )
    RETURN INTEGER IS

  BEGIN

    IF value>upper_limit THEN
      RETURN upper_limit;
    ELSIF value<lower_limit THEN
      RETURN lower_limit;
    ELSE
      RETURN value;
    END IF;

  END range_check;

-- ------------------------------------------------------------------------ --

  FUNCTION check_coefs( coeficients : coeficient_list; number_of_coeficient_bits : INTEGER )
    RETURN coeficient_list IS

    CONSTANT max : INTEGER := (2**(number_of_coeficient_bits-1))-1;
    CONSTANT min : INTEGER := -(2**(number_of_coeficient_bits-1));

    VARIABLE new_coeficients : coeficient_list;

  BEGIN

    FOR i IN coeficients'RANGE LOOP
      new_coeficients(i) := range_check(coeficients(i),min,max);
    END LOOP;

    RETURN new_coeficients;

  END check_coefs;

END sdafirbehv_pack;

-- ------------------------------------------------------------------------ --
-- Single Channel Behavioural Model                                         --
-- ------------------------------------------------------------------------ --

LIBRARY ieee;
USE ieee.std_logic_1164.ALL;
USE ieee.std_logic_arith.ALL;
USE ieee.std_logic_signed.ALL;
-- USE ieee.std_logic_unsigned.ALL;

LIBRARY WORK;
USE WORK.sdafirbehv_pack.ALL;

ENTITY sectionb IS
  PORT( data : IN  std_logic_vector( 8-1 DOWNTO 0 );
          nd : IN  std_logic;
         rfd : OUT std_logic;
        sinf : IN  std_logic;    -- Ignored by behv model
        sinr : IN  std_logic;    -- Ignored by behv model
       soutf : OUT std_logic;    -- Ignored by behv model
       soutr : OUT std_logic;    -- Ignored by behv model
          ck : IN  std_logic;
        rslt : OUT std_logic_vector( 20-1 DOWNTO 0 );
         rdy : OUT std_logic );
END sectionb;

ARCHITECTURE behv OF sectionb IS

-- ------------------------------------------------------------------------ --
-- GUI CONSTANT DECLARATIONS                                                --
-- ------------------------------------------------------------------------ --

  CONSTANT ipdatawidth  : INTEGER := 8;
  CONSTANT opdatawidth  : INTEGER := 20;
  CONSTANT numoftaps    : INTEGER := 10;
  CONSTANT coefwidth    : INTEGER := 8; 
  CONSTANT symmetric    : BOOLEAN := false;
  CONSTANT shortwide    : BOOLEAN := false;
  CONSTANT signed       : BOOLEAN := true;
  CONSTANT cascade      : BOOLEAN := true;
  CONSTANT antisymmetry : BOOLEAN := false;
  CONSTANT coeficients  : coeficient_list := ( 0 => 11,
                                               1 => 12,
                                               2 => 13,
                                               3 => 14,
                                               4 => 15,
                                               5 => 16,
                                               6 => 17,
                                               7 => 18,
                                               8 => 19,
                                               9 => 20,
                                               10 => 0,
                                               11 => 0,
                                               12 => 0,
                                               13 => 0,
                                               14 => 0,
                                               15 => 0,
                                               16 => 0,
                                               17 => 0,
                                               18 => 0,
                                               19 => 0,
                                               20 => 0,
                                               21 => 0,
                                               22 => 0,
                                               23 => 0,
                                               24 => 0,
                                               25 => 0,
                                               26 => 0,
                                               27 => 0,
                                               28 => 0,
                                               29 => 0,
                                               30 => 0,
                                               31 => 0,
                                               32 => 0,
                                               33 => 0,
                                               34 => 0,
                                               35 => 0,
                                               36 => 0,
                                               37 => 0,
                                               38 => 0,
                                               39 => 0 );

-- ------------------------------------------------------------------------ --
-- TYPE DECLARATIONS                                                        --
-- ------------------------------------------------------------------------ --

  TYPE filter_coefficients_type IS ARRAY(1 TO numoftaps) OF INTEGER;

-- ------------------------------------------------------------------------ --
-- FUNCTIONS                                                                --
-- ------------------------------------------------------------------------ --

  FUNCTION bit_reverse( a : std_logic_vector )
    RETURN std_logic_vector IS

    VARIABLE reversed : std_logic_vector(a'REVERSE_RANGE);

  BEGIN

    FOR i IN a'REVERSE_RANGE LOOP
      reversed(i) := a(i);
    END LOOP;

    RETURN reversed;

  END bit_reverse;

-- ------------------------------------------------------------------------ --

 FUNCTION assign_filter_coefficients( coefficients : coeficient_list; symmetric, antisymmetry : BOOLEAN; numoftaps : INTEGER )
    RETURN filter_coefficients_type IS

    VARIABLE filter_coefficients : filter_coefficients_type;
    CONSTANT halfwaypoint        : INTEGER := div2roundup(numoftaps);

  BEGIN
    IF NOT symmetric THEN
      filter_coefficients := filter_coefficients_type(coefficients(0 TO numoftaps-1));
    ELSE
      FOR i IN 1 TO halfwaypoint LOOP
        filter_coefficients(i) := coefficients(i-1);
      END LOOP;
      IF NOT antisymmetry THEN
        FOR i IN halfwaypoint+1 TO numoftaps LOOP
          filter_coefficients(i) := coefficients(numoftaps-i);
        END LOOP;
      ELSE
        FOR i IN halfwaypoint+1 TO numoftaps LOOP
          filter_coefficients(i) := -coefficients(numoftaps-i);
        END LOOP;
      END IF;
    END IF;
    RETURN filter_coefficients;
  END assign_filter_coefficients;

-- ------------------------------------------------------------------------ --
-- CONSTANT DECLARATIONS                                                    --
-- ------------------------------------------------------------------------ --

  CONSTANT filter_coefficients              : filter_coefficients_type := assign_filter_coefficients(check_coefs(coeficients,coefwidth),symmetric,antisymmetry,numoftaps);
  CONSTANT numoftaps_minus_one            : INTEGER                  := numoftaps - 1;

  CONSTANT adder_tree_growth           : INTEGER := growth_table(eval(symmetric))(numoftaps) + eval(symmetric AND shortwide AND numoftaps>26 AND numoftaps<31);
  CONSTANT full_result_width           : INTEGER := coefwidth + ipdatawidth + adder_tree_growth;
  CONSTANT full_result_width_minus_one : INTEGER := full_result_width - 1;

  CONSTANT ipdatawidth_minus_one : INTEGER := ipdatawidth - 1;
  CONSTANT ipdatawidth_minus_two : INTEGER := ipdatawidth - 2;

  CONSTANT latency                      : INTEGER := latency_table(eval(symmetric))(numoftaps);
  CONSTANT latency_minus_one            : INTEGER := latency - 1;
  CONSTANT cycles_per_data_sample       : INTEGER := ipdatawidth + eval(symmetric);
  CONSTANT delay_plus_latency           : INTEGER := cycles_per_data_sample + latency_minus_one;
  CONSTANT delay_plus_latency_minus_one : INTEGER := delay_plus_latency - 1;

  CONSTANT halfnumoftaps           : INTEGER := div2roundup(numoftaps);
  CONSTANT halfnumoftaps_minus_one : INTEGER := halfnumoftaps - 1;
  CONSTANT halfnumoftaps_plus_one  : INTEGER := halfnumoftaps + 1;
  CONSTANT forward_tsb_length      : INTEGER := halfnumoftaps * cycles_per_data_sample;
  CONSTANT reverse_tsb_length      : INTEGER := (numoftaps - halfnumoftaps) * cycles_per_data_sample;

  CONSTANT forward_tsb_length_minus_one : INTEGER := forward_tsb_length - 1;
  CONSTANT forward_tsb_length_minus_two : INTEGER := forward_tsb_length - 2;
  CONSTANT reverse_tsb_length_minus_one : INTEGER := reverse_tsb_length - 1;
  CONSTANT reverse_tsb_length_minus_two : INTEGER := reverse_tsb_length - 2;

  CONSTANT bit_count_max                    : INTEGER                  := cycles_per_data_sample - 2;

-- ------------------------------------------------------------------------ --
-- SIGNAL DECLARATIONS                                                      --
-- ------------------------------------------------------------------------ --

  TYPE   tap_storage_type IS ARRAY(1 TO numoftaps) OF INTEGER;
  SIGNAL tap_storage              : tap_storage_type                         := (OTHERS => 0);

  TYPE pipeline_latency_storage_type IS ARRAY(0 TO delay_plus_latency_minus_one) OF INTEGER;
  SIGNAL pipeline_latency_storage : pipeline_latency_storage_type            := (OTHERS => 0);

  TYPE pipeline_status_storage_type IS ARRAY(0 TO delay_plus_latency_minus_one) OF std_logic;
  SIGNAL pipeline_status_storage  : pipeline_status_storage_type             := (OTHERS => '0');

  SIGNAL forward_tsb_chain        : std_logic_vector(forward_tsb_length_minus_one DOWNTO 0) := (OTHERS => '0');
  SIGNAL reverse_tsb_chain        : std_logic_vector(reverse_tsb_length_minus_one DOWNTO 0) := (OTHERS => '0');
  SIGNAL reverse_tsb_chain_feed   : std_logic := '0';
  SIGNAL forward_tsb_chain_feed   : std_logic := '0';
  SIGNAL psc                      : std_logic_vector(ipdatawidth_minus_one DOWNTO 0) := (OTHERS => '0');

  SIGNAL input_data_bit_counter   : INTEGER                                  := 0;
  SIGNAL new_filter_result        : INTEGER                                  := 0;
  SIGNAL rfd_internal             : std_logic                                := '1';
  SIGNAL rdy_internal             : std_logic                                := '0';
  SIGNAL result_internal          : std_logic_vector(opdatawidth-1 DOWNTO 0) := (OTHERS => 'X');

  SIGNAL shifting_data            : BOOLEAN := FALSE;
  SIGNAL shift_count              : INTEGER := ipdatawidth;

BEGIN

  rfd <= rfd_internal;
  rdy <= rdy_internal;
  rslt <= result_internal WHEN rdy_internal='1' ELSE (OTHERS => 'X');

  reverse_tsb_chain_feed <= rat(sinr) WHEN symmetric AND cascade ELSE forward_tsb_chain(forward_tsb_length_minus_one);
  forward_tsb_chain_feed <= rat(sinf) WHEN cascade ELSE psc(0);
  soutf <= forward_tsb_chain(forward_tsb_length_minus_one) WHEN symmetric ELSE reverse_tsb_chain(reverse_tsb_length_minus_one);
  soutr <= reverse_tsb_chain(reverse_tsb_length_minus_one) WHEN symmetric ELSE 'X';

  psc_and_tsb_delay_lines : PROCESS
  BEGIN

    WAIT UNTIL ck'EVENT AND ck='1';

    IF rfd_internal='0' OR nd='1' THEN
      forward_tsb_chain(0) <= forward_tsb_chain_feed;
      forward_tsb_chain(forward_tsb_length_minus_one DOWNTO 1) <= forward_tsb_chain(forward_tsb_length_minus_two DOWNTO 0);
      reverse_tsb_chain(0) <= reverse_tsb_chain_feed;
      reverse_tsb_chain(reverse_tsb_length_minus_one DOWNTO 1) <= reverse_tsb_chain(reverse_tsb_length_minus_two DOWNTO 0);
    ELSIF rfd_internal='X' THEN
      reverse_tsb_chain <= (OTHERS => 'X');
      forward_tsb_chain <= (OTHERS => 'X');
    END IF;

    IF nd='1' THEN
      psc <= rat_v(data);
    ELSIF nd='X' THEN
      psc <= (OTHERS => 'X');
    ELSE
      psc(ipdatawidth_minus_two DOWNTO 0) <= psc(ipdatawidth_minus_one DOWNTO 1);
      IF signed THEN
        psc(ipdatawidth_minus_one) <= psc(ipdatawidth_minus_one);
      ELSE
        psc(ipdatawidth_minus_one) <= '0';
      END IF;
    END IF;

  END PROCESS; -- psc_and_tsb_delay_lines

-- ------------------------------------------------------------------------ --
-- Keep a history of previous n samples, where n = number of taps           --
-- ------------------------------------------------------------------------ --

  record_sample_data : PROCESS
    VARIABLE input_data_integer     : INTEGER;
    VARIABLE rev_input_data_integer : INTEGER;
    VARIABLE f_shift_data           : std_logic_vector(ipdatawidth-1 DOWNTO 0);
    VARIABLE r_shift_data           : std_logic_vector(ipdatawidth-1 DOWNTO 0);
    CONSTANT last_shift_count       : INTEGER := ipdatawidth;
  BEGIN

    WAIT UNTIL ck'EVENT AND ck='1' AND (nd='1' OR shifting_data);

    IF NOT cascade THEN

      IF signed THEN
        input_data_integer := conv_integer(data); 
      ELSE
        input_data_integer := conv_integer(unsigned(data)); 
      END IF;
      FOR i IN numoftaps_minus_one DOWNTO 1 LOOP
        tap_storage(i+1) <= tap_storage(i);
      END LOOP;
      tap_storage(1) <= input_data_integer;

    ELSE

      IF shifting_data AND nd='0' THEN
        IF shift_count < last_shift_count THEN
          shift_count <= shift_count + 1;
        ELSE
          shifting_data <= FALSE;
        END IF;
      END IF;

      IF shift_count=last_shift_count AND shifting_data THEN
        f_shift_data := bit_reverse(forward_tsb_chain(ipdatawidth_minus_two DOWNTO 0)&forward_tsb_chain_feed);
        r_shift_data := bit_reverse(reverse_tsb_chain(ipdatawidth_minus_two DOWNTO 0)&reverse_tsb_chain_feed);
        IF NOT symmetric THEN
          IF signed THEN
            input_data_integer := conv_integer(f_shift_data); 
          ELSE
            input_data_integer := conv_integer(unsigned(f_shift_data)); 
          END IF;
          FOR i IN numoftaps_minus_one DOWNTO 1 LOOP
            tap_storage(i+1) <= tap_storage(i);
          END LOOP;
          tap_storage(1) <= input_data_integer;
        ELSE
          IF signed THEN
            input_data_integer := conv_integer(f_shift_data); 
            rev_input_data_integer := conv_integer(r_shift_data); 
          ELSE
            input_data_integer := conv_integer(unsigned(f_shift_data)); 
            rev_input_data_integer := conv_integer(unsigned(r_shift_data)); 
          END IF;
          FOR i IN numoftaps_minus_one DOWNTO halfnumoftaps_plus_one LOOP
            tap_storage(i+1) <= tap_storage(i);
          END LOOP;
          tap_storage(halfnumoftaps_plus_one) <= rev_input_data_integer;
          FOR i IN halfnumoftaps_minus_one DOWNTO 1 LOOP
            tap_storage(i+1) <= tap_storage(i);
          END LOOP;
          tap_storage(1) <= input_data_integer;
        END IF;
      END IF;

      IF nd='1' THEN
        shift_count <= 1;
        shifting_data <= TRUE;
      END IF;

    END IF;

  END PROCESS; -- record_sample_data

-- ------------------------------------------------------------------------ --
-- If any of the samples in the history change, calculate the new result    --
-- ------------------------------------------------------------------------ --

  calculate_new_result : PROCESS(tap_storage)
    VARIABLE new_result : INTEGER;
  BEGIN

    new_result := 0;

    FOR i IN 1 TO numoftaps LOOP
      new_result := new_result + filter_coefficients(i)*tap_storage(i);
    END LOOP;

    new_filter_result <= new_result;

  END PROCESS; -- calculate_new_result

-- ------------------------------------------------------------------------ --
-- Use a counter to keep track of which bit is being processed at any time  --
-- ------------------------------------------------------------------------ --

  status_flags : PROCESS
  BEGIN

    WAIT UNTIL ck'EVENT AND ck='1';

    IF nd='1' THEN
      input_data_bit_counter <= 0;
      rfd_internal <= '0';
      pipeline_status_storage(0) <= '1';
    ELSE
      pipeline_status_storage(0) <= '0';
    END IF;
    
    IF input_data_bit_counter=bit_count_max THEN
      rfd_internal <= '1';
      input_data_bit_counter <= 0;
    ELSE
      IF rfd_internal='0' THEN
        input_data_bit_counter <= input_data_bit_counter + 1;
      END IF;
    END IF;

    FOR i IN 1 TO delay_plus_latency_minus_one LOOP
      pipeline_status_storage(i) <= pipeline_status_storage(i-1);
    END LOOP;
    rdy_internal <= pipeline_status_storage(delay_plus_latency_minus_one);

  END PROCESS; -- status_flags

-- ------------------------------------------------------------------------ --
-- Use delaylines to mimic the latency of the real filter - for both the    --
-- result data, and result_ready output                                     --
-- ------------------------------------------------------------------------ --

  pipeline_latency : PROCESS
    VARIABLE full_result : std_logic_vector(full_result_width-1 DOWNTO 0);
  BEGIN

    WAIT UNTIL ck'EVENT AND ck='1';

    pipeline_latency_storage(0) <= new_filter_result;

    FOR i IN 1 TO delay_plus_latency_minus_one LOOP
      pipeline_latency_storage(i) <= pipeline_latency_storage(i-1);
    END LOOP;

    full_result := std_logic_vector(conv_signed(pipeline_latency_storage(delay_plus_latency_minus_one), full_result_width));
    result_internal <= full_result(full_result_width-1 DOWNTO full_result_width-opdatawidth);

  END PROCESS; -- pipeline_latency

END behv;


