-- -------------------------------------------- --
-- Example design illustrating the construction --
-- of larger filters by cascading several smal- --
-- ler filter modules.                          --
--                                              --
-- The composite filter constructed here has an --
-- asymetric impulse response and 29 taps. It   --
-- is constructed using three filter modules;   --
-- two 10-tap and one 9-tap.                    --
-- -------------------------------------------- --

LIBRARY ieee;
USE ieee.std_logic_1164.all;
USE ieee.std_logic_signed.ALL;

-- ------------------------------------------- --
ENTITY fircascadeasm IS
  PORT( data : IN  std_logic_vector(7 DOWNTO 0);
        rslt : OUT std_logic_vector(21 DOWNTO 0);
         rfd : OUT std_logic;
          nd : IN  std_logic;
         rdy : OUT std_logic;
          ck : IN  std_logic );
END fircascadeasm;
-- ------------------------------------------- --

-- ------------------------------------------- --
ARCHITECTURE example OF fircascadeasm IS
-- ------------------------------------------- --

-- ------------------------------------------- --
-- Function declarations                       --
-- ------------------------------------------- --

  FUNCTION sign_extend( a : std_logic_vector;
                     bits : INTEGER)
    RETURN std_logic_vector IS

    VARIABLE extended_a : std_logic_vector(bits-1 DOWNTO 0);

  BEGIN

    extended_a := (OTHERS => a(a'LEFT));
    FOR i IN a'RANGE LOOP
      extended_a(i) := a(i);
    END LOOP; 

    RETURN extended_a;

  END sign_extend;

-- ------------------------------------------- --
-- Signal declarations                         --
-- ------------------------------------------- --

  -- Dangling nets to attach to the three      --
  -- module's unused parallel data-input ports --
  -- and the parallel-to-serial converter's    --
  -- serial input                              --
  SIGNAL data_a_nc, data_b_nc, data_c_nc : std_logic_vector(7 DOWNTO 0);

  -- The parallel output of the parallel-to-   --
  -- serial converter. Only bit 7 is used to   --
  -- sign-extend the serial data               --
  SIGNAL shifted_data              : std_logic_vector(7 DOWNTO 0);

  -- Serial datalinks between cascaded modules --
  SIGNAL linkf_a, linkf_b, linkf_c : std_logic;
  SIGNAL nc_a,    nc_b,    nc_c    : std_logic;

  -- Individual filter's status signals        --
  SIGNAL rfd_a, rfd_b, rfd_c       : std_logic;
  SIGNAL rdy_a, rdy_b, rdy_c       : std_logic;
  SIGNAL delayed_rdy_a             : std_logic;
  SIGNAL double_delayed_rdy_a      : std_logic;

  -- Individual filter's results               --
  SIGNAL rslt_a, rslt_b, rslt_c    : std_logic_vector(19 DOWNTO 0);
  SIGNAL partial_result            : std_logic_vector(20 DOWNTO 0);
  SIGNAL delayed_result_c          : std_logic_vector(19 DOWNTO 0);

  -- Global clock network                      --
  SIGNAL global_ck                 : std_logic;

  -- VCC net                                   --
  SIGNAL vcc                       : std_logic;

-- ------------------------------------------- --
-- Component declarations created by COREGen   --
-- and written into the VHI file. This was     --
-- then cut-and-paste into the design.         --
-- ------------------------------------------- --

-- --------------------------------- --
-- COE File used to create section_a --
-- --------------------------------- --
-- component_name = sectiona;
-- antisymmetry = false;
-- trim_empty_roms = false;
-- short_wide_floorplan = true;
-- input_width = 8;
-- coef_width = 8;
-- number_of_taps = 10;
-- cascade = true;
-- symmetry = false;
-- signed_input_data = true;
-- output_width = 20;
-- coefdata = 1,2,3,4,5,6,7,8,9,10;

  component sectiona port (
	data: IN std_logic_VECTOR(7 downto 0);
	nd: IN std_logic;
	rfd: OUT std_logic;
	sinf: IN std_logic;
	sinr: IN std_logic;
	soutf: OUT std_logic;
	soutr: OUT std_logic;
	ck: IN std_logic;
	rslt: OUT std_logic_VECTOR(19 downto 0);
	rdy: OUT std_logic);
  end component;

-- --------------------------------- --
-- COE File used to create section_b --
-- --------------------------------- --
-- component_name = sectionb;
-- antisymmetry = false;
-- trim_empty_roms = false;
-- short_wide_floorplan = true;
-- input_width = 8;
-- coef_width = 8;
-- number_of_taps = 10;
-- cascade = true;
-- symmetry = false;
-- signed_input_data = true;
-- output_width = 20;
-- coefdata = 11,12,13,14,15,16,17,18,19,20;

  component sectionb port (
	data: IN std_logic_VECTOR(7 downto 0);
	nd: IN std_logic;
	rfd: OUT std_logic;
	sinf: IN std_logic;
	sinr: IN std_logic;
	soutf: OUT std_logic;
	soutr: OUT std_logic;
	ck: IN std_logic;
	rslt: OUT std_logic_VECTOR(19 downto 0);
	rdy: OUT std_logic);
  end component;

-- --------------------------------- --
-- COE File used to create section_c --
-- --------------------------------- --
-- component_name = sectionc;
-- antisymmetry = false;
-- trim_empty_roms = false;
-- short_wide_floorplan = true;
-- input_width = 8;
-- coef_width = 8;
-- number_of_taps = 9;
-- cascade = true;
-- symmetry = false;
-- signed_input_data = true;
-- output_width = 20;
-- coefdata = 21,22,23,24,25,26,27,28,29;

  component sectionc port (
	data: IN std_logic_VECTOR(7 downto 0);
	nd: IN std_logic;
	rfd: OUT std_logic;
	sinf: IN std_logic;
	sinr: IN std_logic;
	soutf: OUT std_logic;
	soutr: OUT std_logic;
	ck: IN std_logic;
	rslt: OUT std_logic_VECTOR(19 downto 0);
	rdy: OUT std_logic);
  end component;

-- ------------------------------------------- --
-- Since all filter modules are created with   --
-- the CASCADE option set we need to serialize --
-- the incoming parallel data before presenting--
-- it to the first filter in the chain         --
-- ------------------------------------------- --

-- ------------------------------------------- --
-- The parallel-to-serial converter below was  --
-- created by COREGen with the following       --
-- parameters:                                 --
--                                             --
-- Port_Width = 8                              --
-- Create_RPM = TRUE                           --
-- Component_Name = serial                     --
-- The resulting VHI file was cut-and-paste in --
-- to this design below.                       --
--                                             --
-- ------------------------------------------- --

  component serial port (
	pi: IN std_logic_VECTOR(7 downto 0);
	sdi: IN std_logic;
	l: IN std_logic;
	c: IN std_logic;
	ce: IN std_logic;
	sdo: OUT std_logic;
	po: OUT std_logic_VECTOR(7 downto 0));
  end component;

-- ------------------------------------------- --
-- Unified library component declarations      --
-- ------------------------------------------- --

  COMPONENT bufg
    PORT ( i : IN  std_logic;
           o : OUT std_logic );
  END COMPONENT;

-- ------------------------------------------- --
BEGIN
-- ------------------------------------------- --

-- ------------------------------------------- --
-- Drive the chip's global clock network with  --
-- the incoming clock                          --
-- ------------------------------------------- --

  u0 : bufg PORT MAP ( i => ck,
                       o => global_ck );

  vcc <= '1';

-- ------------------------------------------- --
-- Component instantiations created by COREGen --
-- and cut-an-paste'd from their respective    --
-- VHI files                                   --
-- ------------------------------------------- --

  u1 : serial port map (
	pi => data,
        -- Sign-extend the shifted data
        -- by repeatedly shifting-in the
        -- MSB of data while data is being
        -- shifted-out LSB first
        -- For unsigned data, attach the
        -- serial data in pin to GND
	sdi => shifted_data(7),
	l => nd,
	c => global_ck,
	ce => vcc,
	sdo => linkf_a,
	po => shifted_data);

  u2 : sectiona port map (
	data => data_a_nc,
	nd => nd,
	rfd => rfd_a,
	sinf => linkf_a,
	sinr => nc_a,
	soutf => linkf_b,
	soutr => OPEN,
	ck => global_ck,
	rslt => rslt_a,
	rdy => rdy_a);

  u3 : sectionb port map (
	data => data_b_nc,
	nd => nd,
	rfd => rfd_b,
	sinf => linkf_b,
	sinr => nc_b,
	soutf => linkf_c,
	soutr => OPEN,
	ck => global_ck,
	rslt => rslt_b,
	rdy => rdy_b);

  u4 : sectionc port map (
	data => data_c_nc,
	nd => nd,
	rfd => rfd_c,
	sinf => linkf_c,
	sinr => nc_c,
	soutf => OPEN,
	soutr => OPEN,
	ck => global_ck,
	rslt => rslt_c,
	rdy => rdy_c);

-- ------------------------------------------- --
-- Connect any module's ready_for_data output  -- 
-- to the overall ready_for_data output since  --
-- all modules should be running in-sync.      --
-- ------------------------------------------- --

  rfd <= rfd_a;

-- ------------------------------------------- --
-- Combine the outputs of all three filters to --
-- form a single result and keep track of the  --
-- result-ready flag                           --
-- ------------------------------------------- --

  clocked : PROCESS( global_ck )
  BEGIN

    IF global_ck'EVENT AND global_ck='1' THEN
        IF rdy_a='1' THEN
        -- Deal with the results
        partial_result       <= sign_extend(rslt_a, 21) +
                                sign_extend(rslt_b, 21);
        delayed_result_c     <= rslt_c;
        rslt                 <= sign_extend(partial_result, 22) +
                                sign_extend(delayed_result_c, 22);
      END IF;
      -- Deal with the flag
      delayed_rdy_a        <= rdy_a;
      double_delayed_rdy_a <= delayed_rdy_a;
      rdy                  <= double_delayed_rdy_a;
    END IF;

  END PROCESS; -- clocked

END example;

 
